const fs = require('fs');

function isPlainObject(value) {
  return value != null && typeof value === 'object' && !Array.isArray(value);
}

const FALLBACK_REMOTE = {
  url: 'https://storage.googleapis.com/cdn.bachmann.xyz/miba-video-manager/posthog-events.json',
  etag: null,
  timeoutMs: 5000
};

const FALLBACK_REGISTRY = {
  version: 1,
  remote: { ...FALLBACK_REMOTE },
  defaults: {
    allow: false,
    propertyAllowlist: ['source', 'registry_version', 'analytics_debug_mode'],
    propertyTransforms: {},
    dropReasons: ['fallback_registry_blocked']
  },
  events: {
    application_start: {
      allow: true,
      propertyAllowlist: ['source', 'app_version', 'platform', 'platform_release', 'timestamp']
    },
    event_schema_download_failed: {
      allow: true,
      propertyAllowlist: ['source', 'attempt', 'reason', 'status_code', 'timestamp']
    },
    event_schema_download_success: {
      allow: true,
      propertyAllowlist: ['source', 'attempt', 'elapsed_ms', 'timestamp']
    },
    event_schema_download_etag_same: {
      allow: true,
      propertyAllowlist: ['source', 'attempt', 'elapsed_ms', 'timestamp']
    },
    new_schema_fetched: {
      allow: true,
      propertyAllowlist: ['source', 'attempt', 'elapsed_ms', 'timestamp']
    },
    event_schema_parse_failed: {
      allow: true,
      propertyAllowlist: ['source', 'reason', 'attempt', 'timestamp']
    },
    '*': {
      allow: false,
      dropReasons: ['fallback_registry_blocked']
    }
  }
};

const SUPPORTED_TRANSFORMS = new Set(['enum', 'truncate']);

function normalizeRemoteMetadata(remote = {}, { fallback = FALLBACK_REMOTE } = {}) {
  const normalized = { ...fallback };
  if (isPlainObject(remote)) {
    if (typeof remote.url === 'string' && remote.url.trim().length > 0) {
      normalized.url = remote.url.trim();
    }
    if (typeof remote.etag === 'string' && remote.etag.trim().length > 0) {
      normalized.etag = remote.etag.trim();
    } else {
      normalized.etag = null;
    }
    const timeout = Number(remote.timeoutMs);
    if (Number.isFinite(timeout) && timeout >= 1000) {
      normalized.timeoutMs = timeout;
    }
  }
  return normalized;
}

function normalizePropertyTransforms(transformConfig = {}) {
  if (!isPlainObject(transformConfig)) {
    return {};
  }
  const normalized = {};
  Object.entries(transformConfig).forEach(([key, value]) => {
    if (!isPlainObject(value)) {
      return;
    }
    const type = value.type;
    if (!SUPPORTED_TRANSFORMS.has(type)) {
      return;
    }
    if (type === 'enum') {
      const values = Array.isArray(value.values) ? value.values.filter((item) => typeof item === 'string') : [];
      if (values.length === 0) {
        return;
      }
      normalized[key] = { type, values };
      return;
    }
    if (type === 'truncate') {
      const maxLength = Number(value.maxLength);
      if (!Number.isInteger(maxLength) || maxLength <= 0 || maxLength > 50_000) {
        return;
      }
      normalized[key] = { type, maxLength };
    }
  });
  return normalized;
}

function normalizeEventRule(eventName, rule, defaults) {
  const normalized = {
    name: eventName,
    allow: defaults.allow,
    propertyAllowlist: Array.isArray(defaults.propertyAllowlist) ? [...defaults.propertyAllowlist] : [],
    propertyTransforms: isPlainObject(defaults.propertyTransforms) ? { ...defaults.propertyTransforms } : {},
    dropReasons: Array.isArray(defaults.dropReasons) ? [...defaults.dropReasons] : []
  };

  if (isPlainObject(rule)) {
    if (typeof rule.allow === 'boolean') {
      normalized.allow = rule.allow;
    }
    if (Array.isArray(rule.propertyAllowlist)) {
      normalized.propertyAllowlist = [
        ...new Set(
          rule.propertyAllowlist
            .concat(defaults.propertyAllowlist || [])
            .filter((prop) => typeof prop === 'string' && prop.trim().length > 0)
        )
      ];
    }
    if (isPlainObject(rule.propertyTransforms)) {
      normalized.propertyTransforms = {
        ...normalized.propertyTransforms,
        ...normalizePropertyTransforms(rule.propertyTransforms)
      };
    }
    if (Array.isArray(rule.dropReasons) && rule.dropReasons.length > 0) {
      normalized.dropReasons = rule.dropReasons.filter((reason) => typeof reason === 'string' && reason.trim().length > 0);
    }
  }

  normalized.propertyAllowlist = normalized.propertyAllowlist.map((name) => name.trim());
  return normalized;
}

function validateAndNormalizeConfig(rawConfig, { source = 'unknown' } = {}) {
  const errors = [];
  const warnings = [];
  if (!isPlainObject(rawConfig)) {
    errors.push('Registry config must be an object.');
    return { config: { ...FALLBACK_REGISTRY }, errors, warnings };
  }
  const version = rawConfig.version;
  if (version !== 1) {
    errors.push(`Unsupported registry version: ${version}.`);
  }

  const defaults = normalizeEventRule(
    '*',
    {
      allow: typeof rawConfig.defaults?.allow === 'boolean' ? rawConfig.defaults.allow : FALLBACK_REGISTRY.defaults.allow,
      propertyAllowlist: Array.isArray(rawConfig.defaults?.propertyAllowlist)
        ? rawConfig.defaults.propertyAllowlist
        : FALLBACK_REGISTRY.defaults.propertyAllowlist,
      propertyTransforms: rawConfig.defaults?.propertyTransforms,
      dropReasons: Array.isArray(rawConfig.defaults?.dropReasons) ? rawConfig.defaults.dropReasons : FALLBACK_REGISTRY.defaults.dropReasons
    },
    FALLBACK_REGISTRY.defaults
  );

  const events = {};
  if (isPlainObject(rawConfig.events)) {
    Object.entries(rawConfig.events).forEach(([eventName, rule]) => {
      if (typeof eventName !== 'string' || eventName.trim().length === 0) {
        warnings.push('Encountered event with invalid name key.');
        return;
      }
      const trimmedName = eventName.trim();
      if (trimmedName !== '*' && !/^[a-z0-9_:.]+$/.test(trimmedName)) {
        warnings.push(`Event name "${trimmedName}" contains invalid characters.`);
      }
      events[trimmedName] = normalizeEventRule(trimmedName, rule, defaults);
    });
  } else {
    warnings.push('Registry config missing "events" object.');
  }

  if (!events['*']) {
    events['*'] = normalizeEventRule('*', {}, defaults);
  }

  const remote = normalizeRemoteMetadata(rawConfig.remote, { fallback: FALLBACK_REMOTE });

  return {
    config: {
      version: 1,
      remote,
      defaults,
      events,
      metadata: {
        source,
        loadedAt: Date.now()
      }
    },
    errors,
    warnings
  };
}

function applyTransforms(propertyName, value, transforms = {}) {
  const transform = transforms[propertyName];
  if (!transform) {
    return { keep: true, value };
  }

  if (transform.type === 'enum') {
    if (typeof value === 'string' && transform.values.includes(value)) {
      return { keep: true, value };
    }
    return { keep: false, reason: 'value_not_in_enum' };
  }

  if (transform.type === 'truncate') {
    if (typeof value === 'string') {
      if (value.length <= transform.maxLength) {
        return { keep: true, value };
      }
      return { keep: true, value: value.slice(0, transform.maxLength), reason: 'value_truncated' };
    }
    if (value == null) {
      return { keep: false, reason: 'value_not_truncatable' };
    }
    const coerced = String(value);
    if (coerced.length <= transform.maxLength) {
      return { keep: true, value: coerced };
    }
    return { keep: true, value: coerced.slice(0, transform.maxLength), reason: 'value_truncated' };
  }

  return { keep: true, value };
}

class AnalyticsEventRegistry {
  constructor({ fallbackConfig = FALLBACK_REGISTRY, logger = console } = {}) {
    this.logger = logger;
    const initial = validateAndNormalizeConfig(fallbackConfig, { source: 'fallback' });
    this.state = {
      config: initial.config,
      warnings: initial.warnings,
      errors: initial.errors
    };
  }

  useFallback(reason) {
    this.logger.warn(`[analytics-registry] Falling back to bundled config: ${reason}`);
    const fallback = validateAndNormalizeConfig(FALLBACK_REGISTRY, { source: 'fallback' });
    this.state = {
      config: fallback.config,
      warnings: fallback.warnings,
      errors: fallback.errors
    };
  }

  update(rawConfig, { source = 'disk', emitWarnings = true } = {}) {
    const { config, errors, warnings } = validateAndNormalizeConfig(rawConfig, { source });
    if (errors.length > 0) {
      this.logger.warn(
        `[analytics-registry] Rejected config from ${source}: ${errors.join('; ')}`
      );
      return false;
    }
    if (emitWarnings && warnings.length > 0) {
      warnings.forEach((warning) => {
        this.logger.warn(`[analytics-registry] Warning (${source}): ${warning}`);
      });
    }
    this.state = { config, warnings, errors };
    return true;
  }

  getSnapshot() {
    const { config, warnings, errors } = this.state;
    return {
      version: config.version,
      remote: { ...config.remote },
      defaults: { ...config.defaults },
      metadata: { ...config.metadata },
      warnings: [...warnings],
      errors: [...errors]
    };
  }

  evaluateEvent(eventName, properties = {}) {
    if (typeof eventName !== 'string' || eventName.trim().length === 0) {
      return {
        allow: false,
        reason: 'invalid_event_name',
        properties: {},
        droppedProperties: {}
      };
    }
    const trimmed = eventName.trim();
    const { events, defaults } = this.state.config;
    const rule = events[trimmed] || events['*'] || defaults;
    const allow = Boolean(rule.allow);

    if (!allow) {
      const reason = rule.dropReasons && rule.dropReasons.length > 0 ? rule.dropReasons[0] : 'event_not_allowed';
      return {
        allow,
        reason,
        properties: {},
        droppedProperties: { '*': reason }
      };
    }

    const propertyAllowlist = Array.isArray(rule.propertyAllowlist) ? rule.propertyAllowlist : [];
    const propertyTransforms = isPlainObject(rule.propertyTransforms) ? rule.propertyTransforms : {};
    const sanitized = {};
    const droppedProperties = {};

    propertyAllowlist.forEach((propertyName) => {
      const value = properties[propertyName];
      if (value === undefined) {
        return;
      }
      const result = applyTransforms(propertyName, value, propertyTransforms);
      if (!result.keep) {
        droppedProperties[propertyName] = result.reason || 'transform_rejected';
        return;
      }
      sanitized[propertyName] = result.value;
      if (result.reason) {
        droppedProperties[propertyName] = result.reason;
      }
    });

    Object.keys(properties).forEach((propertyName) => {
      if (!propertyAllowlist.includes(propertyName)) {
        droppedProperties[propertyName] = 'property_not_allowlisted';
      }
    });

    return {
      allow,
      reason: null,
      properties: sanitized,
      droppedProperties,
      rule
    };
  }
}

function loadRegistryFromFile(filePath, { source = 'disk' } = {}) {
  try {
    const payload = fs.readFileSync(filePath, 'utf8');
    const raw = JSON.parse(payload);
    const { config, errors, warnings } = validateAndNormalizeConfig(raw, { source });
    return {
      success: errors.length === 0,
      raw,
      config,
      errors,
      warnings
    };
  } catch (error) {
    return {
      success: false,
      error
    };
  }
}

module.exports = {
  AnalyticsEventRegistry,
  FALLBACK_REGISTRY,
  validateAndNormalizeConfig,
  normalizeRemoteMetadata,
  loadRegistryFromFile
};


